dofile(LockOn_Options.script_path.."TNL3100/TNL3100_typedefs.lua")

local Terrain = require("terrain")
local sensor_data = get_base_data()
local TNL3100_State = {}
TNL3100_State.__index = TNL3100_State


local CURRENT_HEADING = get_param_handle("CURRENT_HDG")
local GROUND_SPEED_KTS = get_param_handle("GROUND_SPEED_KTS")

function TNL3100_State.new()
    return setmetatable({
        user_waypoints = OrderedList:new(),
        airports = OrderedList:new(),
        actual_waypoint = 2, ----2 
        flight_plan = {},
        temp_wp = {},
        wp_id = 1,
        bearing_wp =0,
        distance = 1,
        eta = 1,
        crosstrack_dist = -1,
        own_pos = {x=0,y=0,alt=0},
        waiting_for_wpt = false,
        entered_a_wp = false,
        entered_wp = 0,
        ui = 
        {
            page = TNL_PAGES.WPT,
        }
    }, TNL3100_State)
end

function TNL3100_State:add_user_waypoint(wp)
    self.user_waypoints:add(1000+self.wp_id,wp)
    self.wp_id = self.wp_id+1
end

function TNL3100_State:add_selected_wp(index)
    if self.entered_a_wp then 
        table.insert(self.flight_plan, index,self.entered_wp)
        self.entered_a_wp = false
    end
end

local function reversed(t)
  local r = {}
  for i = #t, 1, -1 do
    r[#r + 1] = t[i]
  end
  return r
end

function TNL3100_State:reverse_flightplan()
    self.flight_plan = reversed(self.flight_plan)
    self:refresh_flightplans()
end

function TNL3100_State:replace_wp(index)
    if self.entered_a_wp then 
        self.flight_plan[index]= self.entered_wp
        self.entered_a_wp = false
    end
    self:refresh_flightplans()
end

function TNL3100_State:remove_wp(index)
    table.remove(self.flight_plan, index)
    if self.actual_waypoint > #self.flight_plan then
        self.actual_waypoint = #self.flight_plan
    end
    self:refresh_flightplans()
end

function TNL3100_State:refresh_flightplans()
    i = 1
    while i<=#self.flight_plan do
        local id = self.flight_plan[i]
        if id >1000 then 
            if not self.user_waypoints:has(id) then
                --flight_plan[i]= nil
                table.remove(self.flight_plan, i)
            else
                i=i+1
            end
        else
            --airports don't change
        end
    end
    if self.actual_waypoint>#self.flight_plan then
        self.actual_waypoint = #self.flight_plan
    end
end

function TNL3100_State:delete_wp()
    if self.user_waypoints:N() >1 then
        self.user_waypoints:remove()
        self:refresh_flightplans()
    end
    self.user_waypoints:goToFirst()
end

function TNL3100_State:load_waypoints()
	local route = get_mission_route()
	for i = 1, #route, 1 do
        wp = {point ={x = route[i].x, y = route[i].y}, alt = route[i].alt * 3.28084}
        wp.name = route[i].name or ("WP" .. i - 1)
        local lat, lon = Terrain.convertMetersToLatLon(route[i].x, route[i].y)
        wp.point.lat = lat
        wp.point.lon = lon
        
        self.user_waypoints:add(1000+self.wp_id,wp)
        self.wp_id=self.wp_id+1
        self.flight_plan[i]= 1000+i
	end
end

function TNL3100_State:update_position() 
    local x, y, z = sensor_data.getSelfCoordinates()
    self.own_pos.x = x
    self.own_pos.y = z
    self.own_pos.alt = y
end


function TNL3100_State:get_waypoint_by_id(id)
    if id~= nil then 
        if id >1000 then 
            wp = self.user_waypoints:getById(id)
        else
            wp = self.airports:getById(id)
        end
        return wp
    else
        return nil
    end
end

function TNL3100_State:get_flight_plan_waypoints()
    local wps = {}
    for i = 1, #self.flight_plan, 1 do
        wps[i] = self:get_waypoint_by_id(self.flight_plan[i])
	end
    return wps
end

function TNL3100_State:check_actual_waypoint()
    if #self.flight_plan > 0 then 
        if self.actual_waypoint ~=nil and self:get_waypoint_by_id(self.flight_plan[self.actual_waypoint]) ~= nil then 
            local dest = self:get_waypoint_by_id(self.flight_plan[self.actual_waypoint])
            local delta_x = dest.point.x-self.own_pos.x
            local delta_z = dest.point.y-self.own_pos.y
            self.distance = math.sqrt(delta_x*delta_x+delta_z*delta_z)
            if self.distance <300 then
                self.actual_waypoint = self.actual_waypoint + 1
                if self.actual_waypoint>#self.flight_plan then 
                    self.actual_waypoint=#self.flight_plan
                end
            end
        end
    end
end

function TNL3100_State:updateNavParameter()
    if #self.flight_plan > 0 then 
        if self.actual_waypoint ~=nil and self:get_waypoint_by_id(self.flight_plan[self.actual_waypoint]) ~= nil then 
            local dest = self:get_waypoint_by_id(self.flight_plan[self.actual_waypoint])
            local delta_x = dest.point.x-self.own_pos.x
            local delta_z = dest.point.y-self.own_pos.y
            self.distance = math.sqrt(delta_x*delta_x+delta_z*delta_z)
            self.bearing_wp = math.deg(math.atan2(delta_z, delta_x))
            local bearing = CURRENT_HEADING:get()
            local speed = GROUND_SPEED_KTS:get()*0.514444 --m/s
            local ex = math.cos(math.rad(self.bearing_wp-bearing))
            --print_message_to_user(ex)
            local speed_eff = speed*ex
            if speed_eff > 0.1 then 
                self.eta = self.distance/speed_eff --seconds
                
            else
                self.eta = -1
            end

            if self.actual_waypoint >=2 then 
                local prev_wp = self:get_waypoint_by_id(self.flight_plan[self.actual_waypoint-1])
                local delta_wp_x = dest.point.x-prev_wp.point.x
                local delta_wp_y = dest.point.y-prev_wp.point.y
                local dist_wp = math.sqrt(delta_wp_x*delta_wp_x+delta_wp_y*delta_wp_y)
                if dist_wp ==0 then 
                    self.crosstrack_dist = 0.0
                else
                    local nx = delta_wp_x/dist_wp
                    local ny = delta_wp_y/dist_wp
                    self.crosstrack_dist = ny*(self.own_pos.x-prev_wp.point.x)-nx*(self.own_pos.y-prev_wp.point.y)
                end
            else
                self.crosstrack_dist = -1
            end
        else
            self.distance =-1
            self.bearing_wp = -1
        end
    end
end


function TNL3100_State:updateWPFromLatLon(wp)
    x,y = Terrain.convertLatLonToMeters(wp.point.lat,wp.point.lon)
    wp.point.x=x
    wp.point.y=y
end


function TNL3100_State:createWPHere()
    
    local lat, lon = Terrain.convertMetersToLatLon(self.own_pos.x, self.own_pos.y)
    wp = {}
    wp.name = "XXXX"
    wp.point = {x = self.own_pos.x, y = self.own_pos.y, lon = lon,lat = lat}
    wp.alt = self.own_pos.alt
    return wp
end

function TNL3100_State:next_wp(selection)
    if selection == WPT_TYPES.airport then
        self.airports:next()
    else
        self.user_waypoints:next()
    end
end

function TNL3100_State:previous_wp(selection)
    if selection == WPT_TYPES.airport then
        self.airports:prev()
    else
        self.user_waypoints:prev()
    end
end






function TNL3100_State:get_flight_plan()
    return self.flight_plan
end



function TNL3100_State:get_waypoints()
    return self.waypoints
end
function TNL3100_State:get_actual_waypoint()
    return self.actual_waypoint
end

function TNL3100_State:set_actual_waypoint(value)
    self.actual_waypoint = value
end

function TNL3100_State:get_eta()
    return self.eta
end

function TNL3100_State:get_crosstrack_distance()
    return self.crosstrack_dist
end

function TNL3100_State:get_distance()
    return self.distance
end

function TNL3100_State:get_bearing()
    return self.bearing_wp
end

return TNL3100_State